/*
 * Decompiled with CFR 0.152.
 */
package org.jquantlib.testsuite.instruments;

import java.util.HashMap;
import java.util.Map;
import org.jquantlib.QL;
import org.jquantlib.Settings;
import org.jquantlib.daycounters.Actual360;
import org.jquantlib.daycounters.DayCounter;
import org.jquantlib.exercise.EuropeanExercise;
import org.jquantlib.exercise.Exercise;
import org.jquantlib.instruments.AssetOrNothingPayoff;
import org.jquantlib.instruments.CashOrNothingPayoff;
import org.jquantlib.instruments.EuropeanOption;
import org.jquantlib.instruments.GapPayoff;
import org.jquantlib.instruments.OneAssetOption;
import org.jquantlib.instruments.Option;
import org.jquantlib.instruments.PlainVanillaPayoff;
import org.jquantlib.instruments.StrikedTypePayoff;
import org.jquantlib.instruments.VanillaOption;
import org.jquantlib.methods.lattices.AdditiveEQPBinomialTree;
import org.jquantlib.methods.lattices.CoxRossRubinstein;
import org.jquantlib.methods.lattices.JarrowRudd;
import org.jquantlib.methods.lattices.Joshi4;
import org.jquantlib.methods.lattices.LeisenReimer;
import org.jquantlib.methods.lattices.Tian;
import org.jquantlib.methods.lattices.Trigeorgis;
import org.jquantlib.pricingengines.AnalyticEuropeanEngine;
import org.jquantlib.pricingengines.vanilla.BinomialVanillaEngine;
import org.jquantlib.pricingengines.vanilla.IntegralEngine;
import org.jquantlib.pricingengines.vanilla.finitedifferences.FDEuropeanEngine;
import org.jquantlib.processes.BlackScholesMertonProcess;
import org.jquantlib.processes.GeneralizedBlackScholesProcess;
import org.jquantlib.quotes.Handle;
import org.jquantlib.quotes.Quote;
import org.jquantlib.quotes.SimpleQuote;
import org.jquantlib.termstructures.BlackVolTermStructure;
import org.jquantlib.termstructures.YieldTermStructure;
import org.jquantlib.testsuite.util.Flag;
import org.jquantlib.testsuite.util.Utilities;
import org.jquantlib.time.Date;
import org.jquantlib.time.Period;
import org.junit.Assert;
import org.junit.Test;

public class EuropeanOptionTest {
    public EuropeanOptionTest() {
        QL.info("::::: " + this.getClass().getSimpleName() + " :::::");
    }

    private GeneralizedBlackScholesProcess makeProcess(Quote u, YieldTermStructure q, YieldTermStructure r, BlackVolTermStructure vol) {
        return new BlackScholesMertonProcess(new Handle<Quote>(u), new Handle<YieldTermStructure>(q), new Handle<YieldTermStructure>(r), new Handle<BlackVolTermStructure>(vol));
    }

    private VanillaOption makeOption(StrikedTypePayoff payoff, Exercise exercise, SimpleQuote u, YieldTermStructure q, YieldTermStructure r, BlackVolTermStructure vol, EngineType engineType, int binomialSteps, int samples) {
        OneAssetOption.EngineImpl engine;
        GeneralizedBlackScholesProcess stochProcess = this.makeProcess(u, q, r, vol);
        switch (engineType) {
            case Analytic: {
                engine = new AnalyticEuropeanEngine(stochProcess);
                break;
            }
            case JR: {
                engine = new BinomialVanillaEngine(JarrowRudd.class, stochProcess, binomialSteps);
                break;
            }
            case CRR: {
                engine = new BinomialVanillaEngine(CoxRossRubinstein.class, stochProcess, binomialSteps);
                break;
            }
            case EQP: {
                engine = new BinomialVanillaEngine(AdditiveEQPBinomialTree.class, stochProcess, binomialSteps);
                break;
            }
            case TGEO: {
                engine = new BinomialVanillaEngine(Trigeorgis.class, stochProcess, binomialSteps);
                break;
            }
            case TIAN: {
                engine = new BinomialVanillaEngine(Tian.class, stochProcess, binomialSteps);
                break;
            }
            case LR: {
                engine = new BinomialVanillaEngine(LeisenReimer.class, stochProcess, binomialSteps);
                break;
            }
            case JOSHI: {
                engine = new BinomialVanillaEngine(Joshi4.class, stochProcess, binomialSteps);
                break;
            }
            case FiniteDifferences: {
                engine = new FDEuropeanEngine(stochProcess, binomialSteps, samples);
                break;
            }
            case Integral: {
                engine = new IntegralEngine(stochProcess);
                break;
            }
            default: {
                throw new UnsupportedOperationException("unknown engine type: " + (Object)((Object)engineType));
            }
        }
        EuropeanOption option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        return option;
    }

    private int timeToDays(double t) {
        return (int)(t * 360.0 + 0.5);
    }

    @Test
    public void testValues() {
        QL.info("Testing European option values...");
        EuropeanOptionData[] values = new EuropeanOptionData[]{new EuropeanOptionData(Option.Type.Call, 65.0, 60.0, 0.0, 0.08, 0.25, 0.3, 2.1334, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 95.0, 100.0, 0.05, 0.1, 0.5, 0.2, 2.4648, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 19.0, 19.0, 0.1, 0.1, 0.75, 0.28, 1.7011, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 19.0, 19.0, 0.1, 0.1, 0.75, 0.28, 1.7011, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 1.6, 1.56, 0.08, 0.06, 0.5, 0.12, 0.0291, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 70.0, 75.0, 0.05, 0.1, 0.5, 0.35, 4.087, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 90.0, 0.1, 0.1, 0.1, 0.15, 0.0205, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 100.0, 0.1, 0.1, 0.1, 0.15, 1.8734, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 110.0, 0.1, 0.1, 0.1, 0.15, 9.9413, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 90.0, 0.1, 0.1, 0.1, 0.25, 0.315, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 100.0, 0.1, 0.1, 0.1, 0.25, 3.1217, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 110.0, 0.1, 0.1, 0.1, 0.25, 10.3556, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 90.0, 0.1, 0.1, 0.1, 0.35, 0.9474, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 100.0, 0.1, 0.1, 0.1, 0.35, 4.3693, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 110.0, 0.1, 0.1, 0.1, 0.35, 11.1381, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 90.0, 0.1, 0.1, 0.5, 0.15, 0.8069, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 100.0, 0.1, 0.1, 0.5, 0.15, 4.0232, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 110.0, 0.1, 0.1, 0.5, 0.15, 10.5769, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 90.0, 0.1, 0.1, 0.5, 0.25, 2.7026, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 100.0, 0.1, 0.1, 0.5, 0.25, 6.6997, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 110.0, 0.1, 0.1, 0.5, 0.25, 12.7857, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 90.0, 0.1, 0.1, 0.5, 0.35, 4.9329, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 100.0, 0.1, 0.1, 0.5, 0.35, 9.3679, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 100.0, 110.0, 0.1, 0.1, 0.5, 0.35, 15.3086, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 90.0, 0.1, 0.1, 0.1, 0.15, 9.921, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 100.0, 0.1, 0.1, 0.1, 0.15, 1.8734, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 110.0, 0.1, 0.1, 0.1, 0.15, 0.0408, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 90.0, 0.1, 0.1, 0.1, 0.25, 10.2155, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 100.0, 0.1, 0.1, 0.1, 0.25, 3.1217, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 110.0, 0.1, 0.1, 0.1, 0.25, 0.4551, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 90.0, 0.1, 0.1, 0.1, 0.35, 10.8479, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 100.0, 0.1, 0.1, 0.1, 0.35, 4.3693, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 110.0, 0.1, 0.1, 0.1, 0.35, 1.2376, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 90.0, 0.1, 0.1, 0.5, 0.15, 10.3192, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 100.0, 0.1, 0.1, 0.5, 0.15, 4.0232, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 110.0, 0.1, 0.1, 0.5, 0.15, 1.0646, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 90.0, 0.1, 0.1, 0.5, 0.25, 12.2149, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 100.0, 0.1, 0.1, 0.5, 0.25, 6.6997, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 110.0, 0.1, 0.1, 0.5, 0.25, 3.2734, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 90.0, 0.1, 0.1, 0.5, 0.35, 14.4452, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 100.0, 0.1, 0.1, 0.5, 0.35, 9.3679, 1.0E-4), new EuropeanOptionData(Option.Type.Put, 100.0, 110.0, 0.1, 0.1, 0.5, 0.35, 5.7963, 1.0E-4), new EuropeanOptionData(Option.Type.Call, 40.0, 42.0, 0.08, 0.04, 0.75, 0.35, 5.0975, 1.0E-4)};
        Date today = new Settings().evaluationDate();
        Actual360 dc = new Actual360();
        SimpleQuote spot = new SimpleQuote(0.0);
        SimpleQuote qRate = new SimpleQuote(0.0);
        YieldTermStructure qTS = Utilities.flatRate(today, qRate, (DayCounter)dc);
        SimpleQuote rRate = new SimpleQuote(0.0);
        YieldTermStructure rTS = Utilities.flatRate(today, rRate, (DayCounter)dc);
        SimpleQuote vol = new SimpleQuote(0.0);
        BlackVolTermStructure volTS = Utilities.flatVol(today, vol, (DayCounter)dc);
        for (int i = 0; i < values.length - 1; ++i) {
            QL.debug(values[i].toString());
            PlainVanillaPayoff payoff = new PlainVanillaPayoff(values[i].type, values[i].strike);
            Date exDate = today.add(this.timeToDays(values[i].t));
            EuropeanExercise exercise = new EuropeanExercise(exDate);
            spot.setValue(values[i].s);
            qRate.setValue(values[i].q);
            rRate.setValue(values[i].r);
            vol.setValue(values[i].v);
            BlackScholesMertonProcess stochProcess = new BlackScholesMertonProcess(new Handle<SimpleQuote>(spot), new Handle<YieldTermStructure>(qTS), new Handle<YieldTermStructure>(rTS), new Handle<BlackVolTermStructure>(volTS));
            AnalyticEuropeanEngine engine = new AnalyticEuropeanEngine(stochProcess);
            EuropeanOption option = new EuropeanOption(payoff, exercise);
            option.setPricingEngine(engine);
            double calculated = option.NPV();
            double error = Math.abs(calculated - values[i].result);
            double tolerance = values[i].tol;
            StringBuilder sb = new StringBuilder();
            sb.append("error ").append(error).append(" .gt. tolerance ").append(tolerance).append('\n');
            sb.append("    calculated ").append(calculated).append('\n');
            sb.append("    type ").append((Object)values[i].type).append('\n');
            sb.append("    strike ").append(values[i].strike).append('\n');
            sb.append("    s ").append(values[i].s).append('\n');
            sb.append("    q ").append(values[i].q).append('\n');
            sb.append("    r ").append(values[i].r).append('\n');
            sb.append("    t ").append(values[i].t).append('\n');
            sb.append("    v ").append(values[i].v).append('\n');
            sb.append("    result ").append(values[i].result).append('\n');
            sb.append("    tol ").append(values[i].tol);
            if (error <= tolerance) {
                QL.info(" error=" + error);
                continue;
            }
            Assert.fail((String)(exercise + " " + (Object)((Object)payoff.optionType()) + " option with " + payoff + " payoff:\n" + "    spot value:       " + values[i].s + "\n" + "    strike:           " + payoff.strike() + "\n" + "    dividend yield:   " + values[i].q + "\n" + "    risk-free rate:   " + values[i].r + "\n" + "    reference date:   " + today + "\n" + "    maturity:         " + values[i].t + "\n" + "    volatility:       " + values[i].v + "\n\n" + "    expected:         " + values[i].result + "\n" + "    calculated:       " + calculated + "\n" + "    error:            " + error + "\n" + "    tolerance:        " + tolerance));
        }
    }

    @Test
    public void testGreekValues() {
        QL.info("Testing European option greek values...");
        EuropeanOptionData[] values = new EuropeanOptionData[]{new EuropeanOptionData(Option.Type.Call, 100.0, 105.0, 0.1, 0.1, 0.5, 0.36, 0.5946, 0.0), new EuropeanOptionData(Option.Type.Put, 100.0, 105.0, 0.1, 0.1, 0.5, 0.36, -0.3566, 0.0), new EuropeanOptionData(Option.Type.Put, 100.0, 105.0, 0.1, 0.1, 0.5, 0.36, -4.8775, 0.0), new EuropeanOptionData(Option.Type.Call, 60.0, 55.0, 0.0, 0.1, 0.75, 0.3, 0.0278, 0.0), new EuropeanOptionData(Option.Type.Put, 60.0, 55.0, 0.0, 0.1, 0.75, 0.3, 0.0278, 0.0), new EuropeanOptionData(Option.Type.Call, 60.0, 55.0, 0.0, 0.1, 0.75, 0.3, 18.9358, 0.0), new EuropeanOptionData(Option.Type.Put, 60.0, 55.0, 0.0, 0.1, 0.75, 0.3, 18.9358, 0.0), new EuropeanOptionData(Option.Type.Put, 405.0, 430.0, 0.05, 0.07, 0.08333333333333333, 0.2, -31.1924, 0.0), new EuropeanOptionData(Option.Type.Put, 405.0, 430.0, 0.05, 0.07, 0.08333333333333333, 0.2, -0.0855, 0.0), new EuropeanOptionData(Option.Type.Call, 75.0, 72.0, 0.0, 0.09, 1.0, 0.19, 38.7325, 0.0), new EuropeanOptionData(Option.Type.Put, 490.0, 500.0, 0.05, 0.08, 0.25, 0.15, 42.2254, 0.0)};
        double tolerance = 1.0E-4;
        Date today = new Settings().evaluationDate();
        Actual360 dc = new Actual360();
        SimpleQuote spot = new SimpleQuote(0.0);
        SimpleQuote qRate = new SimpleQuote(0.0);
        YieldTermStructure qTS = Utilities.flatRate(today, qRate, (DayCounter)dc);
        SimpleQuote rRate = new SimpleQuote(0.0);
        YieldTermStructure rTS = Utilities.flatRate(today, rRate, (DayCounter)dc);
        SimpleQuote vol = new SimpleQuote(0.0);
        BlackVolTermStructure volTS = Utilities.flatVol(today, vol, (DayCounter)dc);
        int i = -1;
        PlainVanillaPayoff payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        Date exDate = today.add(this.timeToDays(values[i].t));
        EuropeanExercise exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        BlackScholesMertonProcess stochProcess = new BlackScholesMertonProcess(new Handle<SimpleQuote>(spot), new Handle<YieldTermStructure>(qTS), new Handle<YieldTermStructure>(rTS), new Handle<BlackVolTermStructure>(volTS));
        AnalyticEuropeanEngine engine = new AnalyticEuropeanEngine(stochProcess);
        EuropeanOption option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        double calculated = option.delta();
        double error = Math.abs(calculated - values[i].result);
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("delta", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.delta();
        error = Math.abs(calculated - values[i].result);
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("delta", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.elasticity();
        error = Math.abs(Math.abs(calculated - values[i].result));
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("elasticity", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.gamma();
        error = Math.abs(Math.abs(calculated - values[i].result));
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("gamma", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.gamma();
        error = Math.abs(Math.abs(calculated - values[i].result));
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("gamma", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.vega();
        error = Math.abs(Math.abs(calculated - values[i].result));
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("vega", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.vega();
        error = Math.abs(Math.abs(calculated - values[i].result));
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("vega", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.theta();
        error = Math.abs(Math.abs(calculated - values[i].result));
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("theta", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.thetaPerDay();
        error = Math.abs(Math.abs(calculated - values[i].result));
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("theta per day", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.rho();
        error = Math.abs(Math.abs(calculated - values[i].result));
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("rho", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
        payoff = new PlainVanillaPayoff(values[++i].type, values[i].strike);
        exDate = today.add(this.timeToDays(values[i].t));
        exercise = new EuropeanExercise(exDate);
        spot.setValue(values[i].s);
        qRate.setValue(values[i].q);
        rRate.setValue(values[i].r);
        vol.setValue(values[i].v);
        option = new EuropeanOption(payoff, exercise);
        option.setPricingEngine(engine);
        calculated = option.dividendRho();
        error = Math.abs(Math.abs(calculated - values[i].result));
        if (error > 1.0E-4) {
            this.REPORT_FAILURE("dividend rho", payoff, exercise, values[i].s, values[i].q, values[i].r, today, values[i].v, values[i].result, calculated, error, 1.0E-4);
        }
    }

    @Test
    public void testGreeks() {
        QL.info("Testing analytic European option greeks...");
        HashMap<String, Double> tolerance = new HashMap<String, Double>();
        tolerance.put("delta", 1.0E-5);
        tolerance.put("gamma", 1.0E-5);
        tolerance.put("theta", 1.0E-5);
        tolerance.put("rho", 1.0E-5);
        tolerance.put("divRho", 1.0E-5);
        tolerance.put("vega", 1.0E-5);
        HashMap<String, Double> expected = new HashMap<String, Double>();
        HashMap<String, Double> calculated = new HashMap<String, Double>();
        Option.Type[] types = new Option.Type[]{Option.Type.Call, Option.Type.Put};
        double[] strikes = new double[]{50.0, 99.5, 100.0, 100.5, 150.0};
        double[] underlyings = new double[]{100.0};
        double[] qRates = new double[]{0.04, 0.05, 0.06};
        double[] rRates = new double[]{0.01, 0.05, 0.15};
        double[] residualTimes = new double[]{1.0, 2.0};
        double[] vols = new double[]{0.11, 0.5, 1.2};
        Actual360 dc = new Actual360();
        new Settings().setEvaluationDate(Date.todaysDate());
        Date today = new Settings().evaluationDate();
        SimpleQuote spot = new SimpleQuote(0.0);
        SimpleQuote qRate = new SimpleQuote(0.0);
        YieldTermStructure qTS = Utilities.flatRate(qRate, (DayCounter)dc);
        SimpleQuote rRate = new SimpleQuote(0.0);
        YieldTermStructure rTS = Utilities.flatRate(rRate, (DayCounter)dc);
        SimpleQuote vol = new SimpleQuote(0.0);
        BlackVolTermStructure volTS = Utilities.flatVol(vol, (DayCounter)dc);
        for (Option.Type type : types) {
            for (double strike : strikes) {
                for (double residualTime : residualTimes) {
                    Date exDate = today.add(this.timeToDays(residualTime));
                    EuropeanExercise exercise = new EuropeanExercise(exDate);
                    for (int kk = 0; kk < 4; ++kk) {
                        StrikedTypePayoff payoff = null;
                        if (kk == 0) {
                            payoff = new PlainVanillaPayoff(type, strike);
                        } else if (kk == 1) {
                            payoff = new CashOrNothingPayoff(type, strike, 100.0);
                        } else if (kk == 2) {
                            payoff = new AssetOrNothingPayoff(type, strike);
                        } else if (kk == 3) {
                            payoff = new GapPayoff(type, strike, 100.0);
                        }
                        BlackScholesMertonProcess stochProcess = new BlackScholesMertonProcess(new Handle<SimpleQuote>(spot), new Handle<YieldTermStructure>(qTS), new Handle<YieldTermStructure>(rTS), new Handle<BlackVolTermStructure>(volTS));
                        AnalyticEuropeanEngine engine = new AnalyticEuropeanEngine(stochProcess);
                        if (payoff == null) {
                            throw new IllegalArgumentException();
                        }
                        EuropeanOption option = new EuropeanOption(payoff, exercise);
                        option.setPricingEngine(engine);
                        for (double u : underlyings) {
                            for (double q : qRates) {
                                for (double r : rRates) {
                                    for (double v : vols) {
                                        spot.setValue(u);
                                        qRate.setValue(q);
                                        rRate.setValue(r);
                                        vol.setValue(v);
                                        double value = option.NPV();
                                        double delta = option.delta();
                                        double gamma = option.gamma();
                                        double theta = option.theta();
                                        double rho = option.rho();
                                        double drho = option.dividendRho();
                                        double vega = option.vega();
                                        calculated.put("delta", delta);
                                        calculated.put("gamma", gamma);
                                        calculated.put("theta", theta);
                                        calculated.put("rho", rho);
                                        calculated.put("divRho", drho);
                                        calculated.put("vega", vega);
                                        if (!(value > spot.value() * 1.0E-5)) continue;
                                        double du = u * 1.0E-4;
                                        spot.setValue(u + du);
                                        double value_p = option.NPV();
                                        double delta_p = option.delta();
                                        spot.setValue(u - du);
                                        double value_m = option.NPV();
                                        double delta_m = option.delta();
                                        spot.setValue(u);
                                        expected.put("delta", (value_p - value_m) / (2.0 * du));
                                        expected.put("gamma", (delta_p - delta_m) / (2.0 * du));
                                        double dr = r * 1.0E-4;
                                        rRate.setValue(r + dr);
                                        value_p = option.NPV();
                                        rRate.setValue(r - dr);
                                        value_m = option.NPV();
                                        rRate.setValue(r);
                                        expected.put("rho", (value_p - value_m) / (2.0 * dr));
                                        double dq = q * 1.0E-4;
                                        qRate.setValue(q + dq);
                                        value_p = option.NPV();
                                        qRate.setValue(q - dq);
                                        value_m = option.NPV();
                                        qRate.setValue(q);
                                        expected.put("divRho", (value_p - value_m) / (2.0 * dq));
                                        double dv = v * 1.0E-4;
                                        vol.setValue(v + dv);
                                        value_p = option.NPV();
                                        vol.setValue(v - dv);
                                        value_m = option.NPV();
                                        vol.setValue(v);
                                        expected.put("vega", (value_p - value_m) / (2.0 * dv));
                                        Date yesterday = today.sub(1);
                                        Date tomorrow = today.add(1);
                                        double dT = dc.yearFraction(yesterday, tomorrow);
                                        new Settings().setEvaluationDate(yesterday);
                                        value_m = option.NPV();
                                        new Settings().setEvaluationDate(tomorrow);
                                        value_p = option.NPV();
                                        new Settings().setEvaluationDate(Date.todaysDate());
                                        expected.put("theta", (value_p - value_m) / dT);
                                        for (Map.Entry it : calculated.entrySet()) {
                                            String greek = (String)it.getKey();
                                            double expct = (Double)expected.get(greek);
                                            double calcl = (Double)calculated.get(greek);
                                            double tol = (Double)tolerance.get(greek);
                                            double error = Utilities.relativeError(expct, calcl, u);
                                            if (!(error > tol)) continue;
                                            this.REPORT_FAILURE(greek, payoff, exercise, u, q, r, today, v, expct, calcl, error, tol);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    @Test
    public void testImpliedVol() {
        QL.info("Testing European option implied volatility...");
        int maxEvaluations = 100;
        double tolerance = 1.0E-6;
        Option.Type[] types = new Option.Type[]{Option.Type.Call, Option.Type.Put};
        double[] strikes = new double[]{90.0, 99.5, 100.0, 100.5, 110.0};
        int[] lengths = new int[]{36, 180, 360, 1080};
        double[] underlyings = new double[]{90.0, 95.0, 99.9, 100.0, 100.1, 105.0, 110.0};
        double[] qRates = new double[]{0.01, 0.05, 0.1};
        double[] rRates = new double[]{0.01, 0.05, 0.1};
        double[] vols = new double[]{0.01, 0.2, 0.3, 0.7, 0.9};
        Actual360 dc = new Actual360();
        Date today = Date.todaysDate();
        SimpleQuote spot = new SimpleQuote(0.0);
        SimpleQuote qRate = new SimpleQuote(0.0);
        YieldTermStructure qTS = Utilities.flatRate(today, qRate, (DayCounter)dc);
        SimpleQuote rRate = new SimpleQuote(0.0);
        YieldTermStructure rTS = Utilities.flatRate(today, rRate, (DayCounter)dc);
        SimpleQuote vol = new SimpleQuote(0.0);
        BlackVolTermStructure volTS = Utilities.flatVol(today, vol, (DayCounter)dc);
        for (Option.Type type : types) {
            for (double strike2 : strikes) {
                for (int length : lengths) {
                    Date exDate = today.add(length);
                    EuropeanExercise exercise = new EuropeanExercise(exDate);
                    PlainVanillaPayoff payoff = new PlainVanillaPayoff(type, strike2);
                    VanillaOption option = this.makeOption(payoff, exercise, spot, qTS, rTS, volTS, EngineType.Analytic, 0, 0);
                    GeneralizedBlackScholesProcess process = this.makeProcess(spot, qTS, rTS, volTS);
                    for (double u : underlyings) {
                        for (double q : qRates) {
                            for (double r : rRates) {
                                for (double v : vols) {
                                    spot.setValue(u);
                                    qRate.setValue(q);
                                    rRate.setValue(r);
                                    vol.setValue(v);
                                    double value = option.NPV();
                                    double implVol = 0.0;
                                    if (value == 0.0) continue;
                                    vol.setValue(v * 0.5);
                                    if (Math.abs(value - option.NPV()) <= 1.0E-12 || !(Math.abs((implVol = option.impliedVolatility(value, process, 1.0E-6, 100)) - v) > 1.0E-6)) continue;
                                    vol.setValue(implVol);
                                    double value2 = option.NPV();
                                    double error = Utilities.relativeError(value, value2, u);
                                    if (!(error > 1.0E-6)) continue;
                                    Assert.fail((String)((Object)((Object)type) + " option :\n" + "    spot value:          " + u + "\n" + "    strike:              " + strike2 + "\n" + "    dividend yield:      " + q + "\n" + "    risk-free rate:      " + r + "\n" + "    maturity:            " + exDate + "\n\n" + "    original volatility: " + v + "\n" + "    price:               " + value + "\n" + "    implied volatility:  " + implVol + "\n" + "    corresponding price: " + value2 + "\n" + "    error:               " + error));
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    @Test
    public void testImpliedVolContainment() {
        SimpleQuote spot;
        QL.info("Testing self-containment of implied volatility calculation... running");
        int maxEvaluations = 100;
        double tolerance = 1.0E-6;
        Date today = new Settings().evaluationDate();
        Actual360 dc = new Actual360();
        SimpleQuote u = spot = new SimpleQuote(100.0);
        SimpleQuote qRate = new SimpleQuote(0.05);
        YieldTermStructure qTS = Utilities.flatRate(today, qRate, (DayCounter)dc);
        SimpleQuote rRate = new SimpleQuote(0.003);
        YieldTermStructure rTS = Utilities.flatRate(today, rRate, (DayCounter)dc);
        SimpleQuote vol = new SimpleQuote(0.2);
        BlackVolTermStructure volTS = Utilities.flatVol(today, vol, (DayCounter)dc);
        Date exerciseDate = today.add(Period.ONE_YEAR_FORWARD);
        EuropeanExercise exercise = new EuropeanExercise(exerciseDate);
        PlainVanillaPayoff payoff = new PlainVanillaPayoff(Option.Type.Call, 100.0);
        BlackScholesMertonProcess stochProcess = new BlackScholesMertonProcess(new Handle<SimpleQuote>(u), new Handle<YieldTermStructure>(qTS), new Handle<YieldTermStructure>(rTS), new Handle<BlackVolTermStructure>(volTS));
        AnalyticEuropeanEngine engine = new AnalyticEuropeanEngine(stochProcess);
        EuropeanOption option1 = new EuropeanOption(payoff, exercise);
        EuropeanOption option2 = new EuropeanOption(payoff, exercise);
        option1.setPricingEngine(engine);
        option2.setPricingEngine(engine);
        double refValue = option2.NPV();
        Flag f = new Flag();
        option2.addObserver(f);
        option1.impliedVolatility(refValue * 1.5, stochProcess, 1.0E-6, 100);
        if (f.isUp()) {
            Assert.fail((String)"implied volatility calculation triggered a change in another instrument");
        }
        option2.recalculate();
        if (Math.abs(option2.NPV() - refValue) >= 1.0E-8) {
            Assert.fail((String)("implied volatility calculation changed the value of another instrument: \nprevious value: " + refValue + "\n" + "current value:  " + option2.NPV()));
        }
        vol.setValue(vol.value() * 1.5);
        if (!f.isUp()) {
            Assert.fail((String)"volatility change not notified");
        }
        if (Math.abs(option2.NPV() - refValue) <= 1.0E-8) {
            Assert.fail((String)"volatility change did not cause the value to change");
        }
    }

    private void testEngineConsistency(EngineType engine, int binomialSteps, int samples, Map<String, Double> tolerance) {
        this.testEngineConsistency(engine, binomialSteps, samples, tolerance, false);
    }

    private void testEngineConsistency(EngineType engine, int binomialSteps, int samples, Map<String, Double> tolerance, boolean testGreeks) {
        HashMap<String, Double> calculated = new HashMap<String, Double>();
        HashMap<String, Double> expected = new HashMap<String, Double>();
        Option.Type[] types = new Option.Type[]{Option.Type.Call, Option.Type.Put};
        double[] strikes = new double[]{75.0, 100.0, 125.0};
        int[] lengths = new int[]{1};
        double[] underlyings = new double[]{100.0};
        double[] qRates = new double[]{0.0, 0.05};
        double[] rRates = new double[]{0.01, 0.05, 0.15};
        double[] vols = new double[]{0.11, 0.5, 1.2};
        Date today = new Settings().evaluationDate();
        Actual360 dc = new Actual360();
        SimpleQuote spot = new SimpleQuote(0.0);
        SimpleQuote qRate = new SimpleQuote(0.0);
        YieldTermStructure qTS = Utilities.flatRate(today, qRate, (DayCounter)dc);
        SimpleQuote rRate = new SimpleQuote(0.0);
        YieldTermStructure rTS = Utilities.flatRate(today, rRate, (DayCounter)dc);
        SimpleQuote vol = new SimpleQuote(0.0);
        BlackVolTermStructure volTS = Utilities.flatVol(today, vol, (DayCounter)dc);
        for (Option.Type type : types) {
            for (double strike3 : strikes) {
                for (int length2 : lengths) {
                    Date exDate = today.add(this.timeToDays(length2));
                    EuropeanExercise exercise = new EuropeanExercise(exDate);
                    PlainVanillaPayoff payoff = new PlainVanillaPayoff(type, strike3);
                    VanillaOption refOption = this.makeOption(payoff, exercise, spot, qTS, rTS, volTS, EngineType.Analytic, 0, 0);
                    VanillaOption option = this.makeOption(payoff, exercise, spot, qTS, rTS, volTS, engine, binomialSteps, samples);
                    for (double u : underlyings) {
                        for (double q : qRates) {
                            for (double r : rRates) {
                                for (double v : vols) {
                                    spot.setValue(u);
                                    qRate.setValue(q);
                                    rRate.setValue(r);
                                    vol.setValue(v);
                                    expected.clear();
                                    calculated.clear();
                                    double refNPV = refOption.NPV();
                                    double optNPV = option.NPV();
                                    expected.put("value", refNPV);
                                    calculated.put("value", optNPV);
                                    if (testGreeks && option.NPV() > spot.value() * 1.0E-5) {
                                        expected.put("delta", refOption.delta());
                                        expected.put("gamma", refOption.gamma());
                                        expected.put("theta", refOption.theta());
                                        calculated.put("delta", option.delta());
                                        calculated.put("gamma", option.gamma());
                                        calculated.put("theta", option.theta());
                                    }
                                    for (Map.Entry entry : calculated.entrySet()) {
                                        String greek = (String)entry.getKey();
                                        double expct = (Double)expected.get(greek);
                                        double calcl = (Double)calculated.get(greek);
                                        double tol = tolerance.get(greek);
                                        double error = Utilities.relativeError(expct, calcl, u);
                                        if (!(error > tol)) continue;
                                        this.REPORT_FAILURE(greek, payoff, exercise, u, q, r, today, v, expct, calcl, error, tol);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    @Test
    public void testJRBinomialEngines() {
        QL.info("Testing JR binomial European engines against analytic results...");
        EngineType engine = EngineType.JR;
        int timeSteps = 251;
        boolean samples = false;
        HashMap<String, Double> relativeTol = new HashMap<String, Double>(4);
        relativeTol.put("value", 0.002);
        relativeTol.put("delta", 0.001);
        relativeTol.put("gamma", 1.0E-4);
        relativeTol.put("theta", 0.03);
        this.testEngineConsistency(engine, 251, 0, relativeTol, true);
    }

    @Test
    public void testCRRBinomialEngines() {
        QL.info("Testing CRR binomial European engines against analytic results...");
        EngineType engine = EngineType.CRR;
        int timeSteps = 501;
        boolean samples = false;
        HashMap<String, Double> relativeTol = new HashMap<String, Double>(4);
        relativeTol.put("value", 0.002);
        relativeTol.put("delta", 0.001);
        relativeTol.put("gamma", 1.0E-4);
        relativeTol.put("theta", 0.03);
        this.testEngineConsistency(engine, 501, 0, relativeTol, true);
    }

    @Test
    public void testEQPBinomialEngines() {
        QL.info("Testing EQP binomial European engines against analytic results...");
        EngineType engine = EngineType.EQP;
        int timeSteps = 501;
        boolean samples = false;
        HashMap<String, Double> relativeTol = new HashMap<String, Double>(4);
        relativeTol.put("value", 0.02);
        relativeTol.put("delta", 0.001);
        relativeTol.put("gamma", 1.0E-4);
        relativeTol.put("theta", 0.03);
        this.testEngineConsistency(engine, 501, 0, relativeTol, true);
    }

    @Test
    public void testTGEOBinomialEngines() {
        QL.info("Testing TGEO binomial European engines against analytic results...");
        EngineType engine = EngineType.TGEO;
        int timeSteps = 251;
        boolean samples = false;
        HashMap<String, Double> relativeTol = new HashMap<String, Double>(4);
        relativeTol.put("value", 0.002);
        relativeTol.put("delta", 0.001);
        relativeTol.put("gamma", 1.0E-4);
        relativeTol.put("theta", 0.03);
        this.testEngineConsistency(engine, 251, 0, relativeTol, true);
    }

    @Test
    public void testTIANBinomialEngines() {
        QL.info("Testing TIAN binomial European engines against analytic results...");
        EngineType engine = EngineType.TIAN;
        int timeSteps = 251;
        boolean samples = false;
        HashMap<String, Double> relativeTol = new HashMap<String, Double>(4);
        relativeTol.put("value", 0.002);
        relativeTol.put("delta", 0.001);
        relativeTol.put("gamma", 1.0E-4);
        relativeTol.put("theta", 0.03);
        this.testEngineConsistency(engine, 251, 0, relativeTol, true);
    }

    @Test
    public void testLRBinomialEngines() {
        QL.info("Testing LR binomial European engines against analytic results...");
        EngineType engine = EngineType.LR;
        int timeSteps = 251;
        boolean samples = false;
        HashMap<String, Double> relativeTol = new HashMap<String, Double>(4);
        relativeTol.put("value", 1.0E-6);
        relativeTol.put("delta", 0.001);
        relativeTol.put("gamma", 1.0E-4);
        relativeTol.put("theta", 0.03);
        this.testEngineConsistency(engine, 251, 0, relativeTol, true);
    }

    @Test
    public void testJOSHIBinomialEngines() {
        QL.info("Testing Joshi binomial European engines against analytic results...");
        EngineType engine = EngineType.JOSHI;
        int timeSteps = 251;
        boolean samples = false;
        HashMap<String, Double> relativeTol = new HashMap<String, Double>(4);
        relativeTol.put("value", 1.0E-7);
        relativeTol.put("delta", 0.001);
        relativeTol.put("gamma", 1.0E-4);
        relativeTol.put("theta", 0.03);
        this.testEngineConsistency(engine, 251, 0, relativeTol, true);
    }

    @Test
    public void testFdEngines() {
        QL.info("Testing finite-differences European engines against analytic results...");
        EngineType engine = EngineType.FiniteDifferences;
        int timeSteps = 300;
        int gridPoints = 300;
        HashMap<String, Double> relativeTol = new HashMap<String, Double>(4);
        relativeTol.put("value", 1.0E-4);
        relativeTol.put("delta", 1.0E-6);
        relativeTol.put("gamma", 1.0E-6);
        relativeTol.put("theta", 1.0E-4);
        this.testEngineConsistency(engine, 300, 300, relativeTol, true);
    }

    @Test
    public void testIntegralEngines() {
        QL.info("Testing integral engines against analytic results...");
        EngineType engine = EngineType.Integral;
        int timeSteps = 300;
        int gridPoints = 300;
        HashMap<String, Double> relativeTol = new HashMap<String, Double>(1);
        relativeTol.put("value", 1.0E-4);
        this.testEngineConsistency(engine, 300, 300, relativeTol);
    }

    private void REPORT_FAILURE(String greekName, StrikedTypePayoff payoff, Exercise exercise, double s, double q, double r, Date today, double v, double expected, double calculated, double error, double tolerance) {
        StringBuilder sb = new StringBuilder();
        sb.append(exercise).append(" ");
        sb.append((Object)payoff.optionType()).append(" option with ");
        sb.append(payoff.getClass().getName() + " payoff:\n");
        sb.append("    spot value:       " + s + "\n");
        sb.append("    strike:           " + payoff.strike() + "\n");
        sb.append("    dividend yield:   " + q + "\n");
        sb.append("    risk-free rate:   " + r + "\n");
        sb.append("    reference date:   " + today + "\n");
        sb.append("    maturity:         " + exercise.lastDate() + "\n");
        sb.append("    volatility:       " + v + "\n\n");
        sb.append("    expected " + greekName + ":   " + expected + "\n");
        sb.append("    calculated " + greekName + ": " + calculated + "\n");
        sb.append("    error:            " + error + "\n");
        sb.append("    tolerance:        " + tolerance);
        Assert.fail((String)sb.toString());
    }

    private static enum EngineType {
        Analytic,
        JR,
        CRR,
        EQP,
        TGEO,
        TIAN,
        LR,
        JOSHI,
        FiniteDifferences,
        Integral,
        PseudoMonteCarlo,
        QuasiMonteCarlo;

    }

    private static class EuropeanOptionData {
        private final Option.Type type;
        private final double strike;
        private final double s;
        private final double q;
        private final double r;
        private final double t;
        private final double v;
        private final double result;
        private final double tol;

        public EuropeanOptionData(Option.Type type, double strike, double s, double q, double r, double t, double v, double result, double tol) {
            this.type = type;
            this.strike = strike;
            this.s = s;
            this.q = q;
            this.r = r;
            this.t = t;
            this.v = v;
            this.result = result;
            this.tol = tol;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append('[');
            sb.append((Object)this.type).append(", ");
            sb.append(this.strike).append(", ");
            sb.append(this.s).append(", ");
            sb.append(this.q).append(", ");
            sb.append(this.r).append(", ");
            sb.append(this.t).append(", ");
            sb.append(this.v).append(", ");
            sb.append(this.result).append(", ");
            sb.append(this.tol);
            sb.append(']');
            return sb.toString();
        }
    }
}

